<?php
require_once '../config/cors.php';
require_once '../config/database.php';
require_once '../utils/helpers.php';

// Extract authorization token
$token = getAuthToken();

// Validate token & Admin check
// Validate token & Admin check
$systemSecret = getenv('SYSTEM_BACKUP_SECRET');
$providedSecret = $_SERVER['HTTP_X_SYSTEM_SECRET'] ?? '';

// Allow if System Secret matches OR if valid Admin Token
if ($systemSecret && $providedSecret === $systemSecret) {
    // Authorized as System
} else {
    $user = getUserFromToken($token);
    if (!$user || !isAdmin($user['id'])) {
        errorResponse('Unauthorized. Admin access required.', 403);
    }
}

$backupsDir = __DIR__ . '/../../database/backups/';
if (!is_dir($backupsDir)) {
    mkdir($backupsDir, 0755, true);
}

// Database file path (defined in database.php)
// If not defined there, fallback or infer
if (!defined('DB_PATH')) {
    define('DB_PATH', __DIR__ . '/../database.db');
}

$action = $_GET['action'] ?? 'list';

try {
    switch ($action) {
        case 'list':
            // Support SQL dumps, SQLite backups (.bak), and Full System Backups (.zip)
            $files = array_merge(
                glob($backupsDir . '*.sql'), 
                glob($backupsDir . '*.bak'),
                glob($backupsDir . '*.zip')
            );
            $backups = [];
            foreach ($files as $file) {
                $backups[] = [
                    'filename' => basename($file),
                    'size' => filesize($file),
                    'created_at' => date('Y-m-d H:i:s', filemtime($file)),
                    'type' => (pathinfo($file, PATHINFO_EXTENSION) === 'zip') ? 'full' : 'db'
                ];
            }
            // Sort by created_at desc
            usort($backups, function($a, $b) {
                return strcmp($b['created_at'], $a['created_at']);
            });
            jsonResponse($backups);
            break;

        case 'create':
            $type = $_GET['type'] ?? 'db';
            
            if ($type === 'full') {
                // Full System Backup (Recursive Project Root)
                if (!class_exists('ZipArchive')) {
                    errorResponse('ZipArchive extension not enabled', 500);
                }

                // Increase limits for large backup
                set_time_limit(600); // 10 minutes
                ini_set('memory_limit', '512M');

                $filename = 'backup_system_' . date('Y-m-d_H-i-s') . '_' . uniqid() . '.zip';
                $filePath = $backupsDir . $filename;
                $zip = new ZipArchive();

                if ($zip->open($filePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
                    errorResponse("Cannot create zip archive", 500);
                }

                $rootPath = realpath(__DIR__ . '/../../');
                
                // Exclusion list (relative paths or directory names)
                $exclusions = [
                    'node_modules',
                    '.git',
                    'dist',
                    'dist-electron',
                    'release',
                    'database/backups', // Don't backup existing backups
                    '.vscode',
                    '.idea',
                    '.DS_Store',
                    'Thumbs.db'
                ];

                /**
                 * Recursive function to add files to zip
                 */
                $files = new RecursiveIteratorIterator(
                    new RecursiveDirectoryIterator($rootPath, RecursiveDirectoryIterator::SKIP_DOTS),
                    RecursiveIteratorIterator::LEAVES_ONLY
                );

                foreach ($files as $name => $file) {
                    // Skip directories (they are added automatically when files are added)
                    if (!$file->isFile()) {
                        continue;
                    }

                    $filePath = $file->getRealPath();
                    $relativePath = substr($filePath, strlen($rootPath) + 1);

                    // Check exclusions
                    $exclude = false;
                    foreach ($exclusions as $excludedItem) {
                        // Check if relative path starts with excluded item directory
                        // OS-agnostic directory separator check
                        if (
                            strpos($relativePath, $excludedItem . DIRECTORY_SEPARATOR) === 0 || 
                            $relativePath === $excludedItem
                        ) {
                            $exclude = true;
                            break;
                        }
                    }

                    if ($exclude) {
                        continue;
                    }

                    // Add current file to archive
                    $zip->addFile($filePath, $relativePath);
                }

                $zip->close();

                jsonResponse(['message' => 'Full system backup created successfully', 'filename' => $filename]);


            } elseif ($type === 'split') {
                // Split Backup Strategy (Native TAR for speed)
                // 1. Core (Source + Config)
                // 2. Assets (Public Uploads)
                // 3. Database

                $timestamp = date('Y-m-d_H-i-s') . '_' . uniqid();
                $rootPath = realpath(__DIR__ . '/../../');
                
                $files = [];

                // --- 1. Core Backup ---
                $coreFilename = 'backup_core_' . $timestamp . '.zip';
                $coreAbsPath = $backupsDir . $coreFilename;
                
                // Construct tar command for Core
                // Exclude: node_modules, .git, dist, database, public (handled separately), backups
                // Note: Windows tar (bsdtar) uses standard syntax
                $cmdCore = sprintf(
                    'tar -a -cf "%s" -C "%s" --exclude "node_modules" --exclude ".git" --exclude "dist" --exclude "dist-electron" --exclude "release" --exclude "database" --exclude "public" .',
                    $coreAbsPath,
                    $rootPath
                );
                
                exec($cmdCore, $outputCore, $returnCore);
                
                if ($returnCore === 0 && file_exists($coreAbsPath)) {
                    $files[] = [
                        'type' => 'core',
                        'filename' => $coreFilename,
                        'size' => filesize($coreAbsPath)
                    ];
                }

                // --- 2. Assets Backup ---
                $assetsFilename = 'backup_assets_' . $timestamp . '.zip';
                $assetsAbsPath = $backupsDir . $assetsFilename;
                
                // Archive 'public' directory
                $cmdAssets = sprintf(
                    'tar -a -cf "%s" -C "%s" "public"',
                    $assetsAbsPath,
                    $rootPath
                );
                
                exec($cmdAssets, $outputAssets, $returnAssets);

                if ($returnAssets === 0 && file_exists($assetsAbsPath)) {
                    $files[] = [
                        'type' => 'assets',
                        'filename' => $assetsFilename,
                        'size' => filesize($assetsAbsPath)
                    ];
                }

                // --- 3. Database Backup ---
                $dbFilename = 'backup_db_' . $timestamp . '.bak';
                $dbAbsPath = $backupsDir . $dbFilename;
                
                try {
                    $stmt = $pdo->prepare("VACUUM INTO ?");
                    $stmt->execute([$dbAbsPath]);
                    $files[] = [
                        'type' => 'db',
                        'filename' => $dbFilename,
                        'size' => filesize($dbAbsPath)
                    ];
                } catch (PDOException $e) {
                    if (copy(DB_PATH, $dbAbsPath)) {
                        $files[] = [
                            'type' => 'db',
                            'filename' => $dbFilename,
                            'size' => filesize($dbAbsPath)
                        ];
                    }
                }

                jsonResponse([
                    'message' => 'Split backup created successfully', 
                    'files' => $files
                ]);

            } else {
                // Standard DB Backup
                $filename = 'backup_db_' . date('Y-m-d_H-i-s') . '_' . uniqid() . '.bak';
                $filePath = $backupsDir . $filename;
                
                try {
                    $stmt = $pdo->prepare("VACUUM INTO ?");
                    $stmt->execute([$filePath]);
                    jsonResponse(['message' => 'Database backup created successfully', 'filename' => $filename]);
                } catch (PDOException $e) {
                    if (copy(DB_PATH, $filePath)) {
                        jsonResponse(['message' => 'Database backup created successfully (Copy)', 'filename' => $filename]);
                    } else {
                        errorResponse('Failed to create backup file', 500);
                    }
                }
            }
            break;

        case 'restore':
            $filename = $_GET['filename'] ?? '';
            $filename = basename($filename); // Security: prevent directory traversal
            $filePath = $backupsDir . $filename;

            if (empty($filename) || !file_exists($filePath) || strpos($filename, 'backup_') !== 0) {
                errorResponse('Invalid backup file', 400);
            }

            try {
                $extension = pathinfo($filename, PATHINFO_EXTENSION);
                
                // Create safety backup of current state before restore
                $safetyBackupName = 'safety_backup_' . date('Y-m-d_H-i-s') . '_' . uniqid() . '.bak';
                $safetyBackupPath = $backupsDir . $safetyBackupName;
                
                if (file_exists(DB_PATH)) {
                    copy(DB_PATH, $safetyBackupPath);
                }

                if ($extension === 'bak' || $extension === 'db') {
                    // Database restore
                    if (!copy($filePath, DB_PATH)) {
                        // Rollback if copy fails
                        if (file_exists($safetyBackupPath)) {
                            copy($safetyBackupPath, DB_PATH);
                        }
                        errorResponse('Failed to restore database backup', 500);
                    }
                    
                    jsonResponse([
                        'message' => 'Database restored successfully',
                        'safety_backup' => $safetyBackupName
                    ]);

                } elseif ($extension === 'zip') {
                    // Full/Split system restore
                    if (!class_exists('ZipArchive')) {
                        errorResponse('ZipArchive extension not enabled', 500);
                    }

                    $zip = new ZipArchive();
                    if ($zip->open($filePath) !== TRUE) {
                        errorResponse('Cannot open backup archive', 500);
                    }

                    $rootPath = realpath(__DIR__ . '/../../');
                    
                    // Extract to root (this will overwrite existing files)
                    if (!$zip->extractTo($rootPath)) {
                        $zip->close();
                        errorResponse('Failed to extract backup archive', 500);
                    }
                    
                    $zip->close();

                    jsonResponse([
                        'message' => 'System restored successfully from archive',
                        'safety_backup' => $safetyBackupName,
                        'note' => 'Please restart the application for changes to take effect'
                    ]);

                } else {
                    errorResponse('Unsupported backup file type', 400);
                }

            } catch (Exception $e) {
                // Attempt rollback
                if (file_exists($safetyBackupPath)) {
                    copy($safetyBackupPath, DB_PATH);
                }
                errorResponse('Restore failed: ' . $e->getMessage(), 500);
            }
            break;

        case 'delete':
            $filename = $_GET['filename'] ?? '';
            // Security: sanitization to prevent directory traversal
            $filename = basename($filename);
            $filePath = $backupsDir . $filename;
            
            if (!empty($filename) && file_exists($filePath) && strpos($filename, 'backup_') === 0) {
                unlink($filePath);
                jsonResponse(['message' => 'Backup deleted successfully']);
            } else {
                errorResponse('Invalid file or file not found', 400);
            }
            break;

        case 'download':
            $filename = $_GET['filename'] ?? '';
            $filename = basename($filename);
            $filePath = $backupsDir . $filename;
            
            if (!empty($filename) && file_exists($filePath) && strpos($filename, 'backup_') === 0) {
                header('Content-Description: File Transfer');
                header('Content-Type: application/octet-stream');
                header('Content-Disposition: attachment; filename="' . $filename . '"');
                header('Expires: 0');
                header('Cache-Control: must-revalidate');
                header('Pragma: public');
                header('Content-Length: ' . filesize($filePath));
                readfile($filePath);
                exit;
            } else {
                errorResponse('Invalid file or file not found', 400);
            }
            break;

        default:
            errorResponse('Invalid action', 400);
    }
} catch (Exception $e) {
    errorResponse('Backup Error: ' . $e->getMessage(), 500);
}
?>
