<?php
require_once '../config/cors.php';
require_once '../config/database.php';
require_once '../utils/helpers.php';

$token = getAuthToken();
$user = getUserFromToken($token);

if (!$user || $user['role'] !== 'admin') {
    errorResponse('Unauthorized access', 403);
}

$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'POST') {
    $data = getJsonInput();

    if (!isset($data['userId']) || !isset($data['symbol']) || !isset($data['amount']) || !isset($data['type'])) {
        errorResponse('Missing required fields');
    }

    $targetUserId = $data['userId'];
    $symbol = strtoupper($data['symbol']);
    $amount = floatval($data['amount']);
    $type = $data['type']; // 'credit' or 'debit'

    if ($amount <= 0) {
        errorResponse('Amount must be positive');
    }

    try {
        $pdo->beginTransaction();

        // Check if user exists
        $stmt = $pdo->prepare("SELECT id FROM users WHERE id = ?");
        $stmt->execute([$targetUserId]);
        if (!$stmt->fetch()) {
            throw new Exception("User not found");
        }

        // Check/Create asset entry
        $stmt = $pdo->prepare("SELECT balance FROM user_assets WHERE user_id = ? AND symbol = ?");
        $stmt->execute([$targetUserId, $symbol]);
        $asset = $stmt->fetch();

        $currentBalance = $asset ? floatval($asset['balance']) : 0.0;
        $newBalance = $currentBalance;

        if ($type === 'credit') {
            $newBalance += $amount;
        } elseif ($type === 'debit') {
            if ($currentBalance < $amount) {
                throw new Exception("Insufficient balance for debit");
            }
            $newBalance -= $amount;
        } else {
            throw new Exception("Invalid transaction type");
        }

        // Upsert asset
        // Simplified upsert for SQLite/MySQL compatibility
        if ($asset) {
            // Update existing
            $stmt = $pdo->prepare("UPDATE user_assets SET balance = ?, name = ?, color = ?, icon = ? WHERE user_id = ? AND symbol = ?");
            $name = $symbol;
            $color = 'from-gray-500 to-gray-600';
            $icon = 'Coins'; // Legacy icon field, kept for compatibility
            $stmt->execute([$newBalance, $name, $color, $icon, $targetUserId, $symbol]);
            
        } else {
            // Insert new
            $stmt = $pdo->prepare("INSERT INTO user_assets (user_id, symbol, name, balance, color, icon) VALUES (?, ?, ?, ?, ?, ?)");
            $name = $symbol; // Default name
            $color = 'from-gray-500 to-gray-600';
            $icon = 'Coins';
            $stmt->execute([$targetUserId, $symbol, $name, $newBalance, $color, $icon]);
        }

        // Log transaction
        $stmt = $pdo->prepare("INSERT INTO admin_logs (id, admin_id, action, target_table, target_id, details) VALUES (?, ?, ?, ?, ?, ?)");
        $details = json_encode([
            'type' => $type,
            'amount' => $amount,
            'symbol' => $symbol,
            'old_balance' => $currentBalance,
            'new_balance' => $newBalance,
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0'
        ]);
        $stmt->execute([
            generateUUID(),
            $user['id'], 
            'wallet_adjustment', 
            'user_assets',
            $targetUserId,
            $details
        ]);

        $pdo->commit();
        jsonResponse(['message' => 'Wallet updated successfully', 'new_balance' => $newBalance]);

    } catch (Exception $e) {
        $pdo->rollBack();
        errorResponse($e->getMessage(), 400);
    }
} else {
    errorResponse('Method not allowed', 405);
}
?>
