<?php
require_once '../config/cors.php';
require_once '../config/database.php';
require_once '../utils/helpers.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Extract authorization token
$token = getAuthToken();

// Validate token
$user = getUserFromToken($token);
if (!$user) {
    errorResponse('User not authenticated', 401);
}

// Check if user is admin
if (!isAdmin($user['id'])) {
    errorResponse('Unauthorized: Admin access required', 403);
}

try {
    // Get total users count
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM users");
    $totalUsers = $stmt->fetch()['total'];

    // Get total users from last month for comparison
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM users WHERE created_at >= datetime('now', '-1 month')");
    $usersLastMonth = $stmt->fetch()['total'];
    
    // Get active licenses count
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM license_keys k JOIN license_status s ON k.status_id = s.id WHERE s.name = 'active'");
    $activeLicenses = $stmt->fetch()['total'];
    
    // Get total licenses from last month for comparison
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM license_keys k JOIN license_status s ON k.status_id = s.id WHERE s.name = 'active' AND k.created_at >= datetime('now', '-1 month')");
    $licensesLastMonth = $stmt->fetch()['total'];
    
    // Calculate revenue from active licenses
    $stmt = $pdo->query("
        SELECT 
            SUM(CASE 
                WHEN t.name = 'basic' THEN 9.99
                WHEN t.name = 'pro' THEN 29.99
                WHEN t.name = 'enterprise' THEN 99.99
                ELSE 0
            END) as revenue
        FROM license_keys k
        JOIN license_status s ON k.status_id = s.id
        JOIN subscription_tiers t ON k.subscription_tier_id = t.id
        WHERE s.name = 'active'
    ");
    $revenue = $stmt->fetch()['revenue'] ?? 0;
    
    // Get revenue from last month
    $stmt = $pdo->query("
        SELECT 
            SUM(CASE 
                WHEN subscription_tier = 'basic' THEN 9.99
                WHEN subscription_tier = 'pro' THEN 29.99
                WHEN subscription_tier = 'enterprise' THEN 99.99
                ELSE 0
            END) as revenue
        FROM licenses 
        WHERE status = 'active' 
        AND created_at >= datetime('now', '-1 month')
    ");
    $revenueLastMonth = $stmt->fetch()['revenue'] ?? 0;
    
    // Get total email subscribers (from mail_activity or similar)
    $stmt = $pdo->query("SELECT COUNT(DISTINCT recipient_email) as total FROM mail_activity");
    $subscribers = $stmt->fetch()['total'];
    
    // Get subscribers from last month
    $stmt = $pdo->query("SELECT COUNT(DISTINCT recipient_email) as total FROM mail_activity WHERE created_at >= datetime('now', '-1 month')");
    $subscribersLastMonth = $stmt->fetch()['total'];
    
    // Calculate percentage changes
    $usersChange = $totalUsers > 0 ? round((($totalUsers - ($totalUsers - $usersLastMonth)) / max($totalUsers - $usersLastMonth, 1)) * 100, 1) : 0;
    $licensesChange = $activeLicenses > 0 ? round((($activeLicenses - ($activeLicenses - $licensesLastMonth)) / max($activeLicenses - $licensesLastMonth, 1)) * 100, 1) : 0;
    $revenueChange = $revenue > 0 ? round((($revenue - ($revenue - $revenueLastMonth)) / max($revenue - $revenueLastMonth, 1)) * 100, 1) : 0;
    $subscribersChange = $subscribers > 0 ? round((($subscribers - ($subscribers - $subscribersLastMonth)) / max($subscribers - $subscribersLastMonth, 1)) * 100, 1) : 0;
    
    // Get revenue trend for last 7 days
    $revenueTrend = [];
    for ($i = 6; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        $stmt = $pdo->prepare("
            SELECT 
                SUM(CASE 
                    WHEN subscription_tier = 'basic' THEN 9.99
                    WHEN subscription_tier = 'pro' THEN 29.99
                    WHEN subscription_tier = 'enterprise' THEN 99.99
                    ELSE 0
                END) as daily_revenue
            FROM licenses 
            WHERE status = 'active' 
            AND DATE(created_at) = ?
        ");
        $stmt->execute([$date]);
        $dailyRevenue = $stmt->fetch()['daily_revenue'] ?? 0;
        
        $revenueTrend[] = [
            'date' => date('M d', strtotime($date)),
            'revenue' => (float)$dailyRevenue
        ];
    }

    // NEW: Get user growth for last 7 days
    $userGrowth = [];
    for ($i = 6; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM users WHERE DATE(created_at) = ?");
        $stmt->execute([$date]);
        $count = $stmt->fetch()['count'] ?? 0;
        $userGrowth[] = [
            'date' => date('M d', strtotime($date)),
            'count' => (int)$count
        ];
    }

    // NEW: Get license activation activity for last 7 days
    $licenseActivity = [];
    for ($i = 6; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM licenses WHERE status = 'active' AND DATE(created_at) = ?");
        $stmt->execute([$date]);
        $count = $stmt->fetch()['count'] ?? 0;
        $licenseActivity[] = [
            'date' => date('M d', strtotime($date)),
            'count' => (int)$count
        ];
    }

    // NEW: Get order status distribution
    $stmt = $pdo->query("SELECT status, COUNT(*) as count FROM orders GROUP BY status");
    $orderDistribution = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $formattedOrderDistribution = [];
    foreach ($orderDistribution as $row) {
        $formattedOrderDistribution[] = [
            'name' => ucfirst($row['status']),
            'value' => (int)$row['count']
        ];
    }

    // NEW: Get total system wallet balance (simplified to a single value for now)
    $stmt = $pdo->query("SELECT SUM(balance) as total FROM user_assets");
    $totalWalletBalance = $stmt->fetch()['total'] ?? 0;
    
    // Get license distribution by tier
    $stmt = $pdo->query("
        SELECT 
            subscription_tier as tier,
            COUNT(*) as count
        FROM licenses 
        WHERE status = 'active'
        GROUP BY subscription_tier
    ");
    $licenseDistribution = $stmt->fetchAll();
    
    // Format for chart
    $chartData = [];
    foreach ($licenseDistribution as $row) {
        $chartData[] = [
            'name' => ucfirst($row['tier']),
            'value' => (int)$row['count']
        ];
    }
    
    // Get recent activity (last 10 activities)
    $stmt = $pdo->query("
        SELECT 
            'User Signup' as type,
            u.email as description,
            u.created_at as timestamp,
            'user' as category
        FROM users u
        ORDER BY u.created_at DESC
        LIMIT 5
    ");
    $userActivity = $stmt->fetchAll();
    
    $stmt = $pdo->query("
        SELECT 
            'License Activated' as type,
            'License ' || substr(key_value, 1, 8) || '...' as description,
            created_at as timestamp,
            'license' as category
        FROM licenses
        ORDER BY created_at DESC
        LIMIT 5
    ");
    $licenseActivityLog = $stmt->fetchAll();
    
    $stmt = $pdo->query("
        SELECT 
            'Email Sent' as type,
            'To: ' || recipient_email as description,
            created_at as timestamp,
            'email' as category
        FROM mail_activity
        WHERE status = 'sent'
        ORDER BY created_at DESC
        LIMIT 5
    ");
    $emailActivity = $stmt->fetchAll();
    
    // Merge and sort all activities by timestamp
    $recentActivity = array_merge($userActivity, $licenseActivityLog, $emailActivity);
    usort($recentActivity, function($a, $b) {
        return strtotime($b['timestamp']) - strtotime($a['timestamp']);
    });
    $recentActivity = array_slice($recentActivity, 0, 10);
    
    // Get system health status
    $systemHealth = [
        'database' => 'operational',
        'api' => 'operational',
        'email_service' => 'operational',
        'backup' => 'scheduled'
    ];
    
    // Try to verify SMTP is configured
    try {
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM smtp_settings WHERE is_active = 1");
        $smtpConfigured = $stmt->fetch()['total'] > 0;
        $systemHealth['email_service'] = $smtpConfigured ? 'operational' : 'not_configured';
    } catch (Exception $e) {
        $systemHealth['email_service'] = 'unknown';
    }
    
    jsonResponse([
        'stats' => [
            'total_users' => $totalUsers,
            'users_change' => $usersChange,
            'active_licenses' => $activeLicenses,
            'licenses_change' => $licensesChange,
            'revenue' => $revenue,
            'revenue_change' => $revenueChange,
            'subscribers' => $subscribers,
            'subscribers_change' => $subscribersChange,
            'total_wallet_balance' => (float)$totalWalletBalance
        ],
        'user_growth' => $userGrowth,
        'license_activity' => $licenseActivity,
        'order_distribution' => $formattedOrderDistribution,
        'revenue_trend' => $revenueTrend,
        'license_distribution' => $chartData,
        'recent_activity' => $recentActivity,
        'system_health' => $systemHealth
    ], 200);
    
} catch (Exception $e) {
    errorResponse('Error fetching admin dashboard data: ' . $e->getMessage(), 500);
}
?>
