<?php
require_once __DIR__ . '/../vendor/autoload.php';

/**
 * CRITICAL: THIS IS A LOCKED CORE COMPONENT.
 * DO NOT MODIFY THIS FILE WITHOUT EXPLICIT AUTHORIZATION.
 * ALL EMAIL LOGIC MUST BE ROUTED THROUGH THIS UTILITY.
 * 
 * Future features must build upon this logic, not replace it.
 */

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

/**
 * Robust email sending function that uses database configuration
 * 
 * @param string $to Recipient email
 * @param string $subject Email subject
 * @param string $body HTML body
 * @param array $options Additional options (template, data, from_email, from_name)
 * @return array ['success' => bool, 'message' => string, 'provider' => string]
 */
function sendRobustEmail($to, $subject, $body, $options = []) {
    global $pdo;
    
    try {
        // 1. Get active email settings
        $stmt = $pdo->prepare("SELECT * FROM smtp_settings WHERE is_active = TRUE LIMIT 1");
        $stmt->execute();
        $settings = $stmt->fetch();

        if (!$settings) {
            return ['success' => false, 'message' => 'No active email settings configured'];
        }

        $provider = $settings['provider'];
        $fromEmail = $options['from_email'] ?? $settings['from_email'];
        $fromName = $options['from_name'] ?? $settings['from_name'];
        
        $mailSent = false;
        $errorMessage = '';

        switch ($provider) {
            case 'resend':
                $mailSent = _sendViaResend($to, $subject, $body, $fromEmail, $fromName, $settings['api_key']);
                break;

            case 'emailjs':
                $mailSent = _sendViaEmailJS($to, $subject, $body, $fromEmail, $fromName, $settings['service_id'], $settings['template_id'], $settings['user_id']);
                break;

            case 'phpmail':
                $headers = "From: $fromName <$fromEmail>\r\n";
                $headers .= "MIME-Version: 1.0\r\n";
                $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
                $mailSent = mail($to, $subject, $body, $headers);
                if (!$mailSent) $errorMessage = 'PHP mail() function failed';
                break;

            default: // SMTP/Gmail/Outlook
                $mail = new PHPMailer(true);
                $mail->isSMTP();
                $mail->Host = $settings['host'];
                $mail->SMTPAuth = true;
                $mail->Username = $settings['username'];
                $mail->Password = $settings['password'];
                
                $port = $settings['port'] ?? 587;
                $mail->SMTPSecure = ($port == 465) ? PHPMailer::ENCRYPTION_SMTPS : PHPMailer::ENCRYPTION_STARTTLS;
                $mail->Port = $port;
                
                $mail->setFrom($fromEmail, $fromName);
                $mail->addAddress($to);
                $mail->isHTML(true);
                $mail->Subject = $subject;
                $mail->Body = $body;
                $mail->AltBody = strip_tags($body);
                
                $mailSent = $mail->send();
                break;
        }

        // 2. Log Activity
        _logMailActivity($to, $subject, $body, $mailSent ? 'sent' : 'failed', $provider, $errorMessage);

        return [
            'success' => $mailSent,
            'message' => $mailSent ? 'Email sent successfully' : ($errorMessage ?: 'Failed to send email'),
            'provider' => $provider
        ];

    } catch (Exception $e) {
        $msg = "Mailer Error: " . $e->getMessage();
        _logMailActivity($to, $subject, $body, 'failed', $provider ?? 'unknown', $msg);
        return ['success' => false, 'message' => $msg];
    }
}

/**
 * Internal helper for Resend API
 */
function _sendViaResend($to, $subject, $body, $fromEmail, $fromName, $apiKey) {
    if (!$apiKey) throw new Exception("Resend API key missing");
    
    $payload = json_encode([
        'from' => "$fromName <$fromEmail>",
        'to' => [$to],
        'subject' => $subject,
        'html' => $body
    ]);
    
    $ch = curl_init('https://api.resend.com/emails');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $apiKey,
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return $httpCode === 200 || $httpCode === 201;
}

/**
 * Internal helper for EmailJS API
 */
function _sendViaEmailJS($to, $subject, $body, $fromEmail, $fromName, $serviceId, $templateId, $userId) {
    $payload = json_encode([
        'service_id' => $serviceId,
        'template_id' => $templateId,
        'user_id' => $userId,
        'template_params' => [
            'to_email' => $to,
            'from_name' => $fromName,
            'from_email' => $fromEmail,
            'subject' => $subject,
            'message_html' => $body
        ]
    ]);
    
    $ch = curl_init('https://api.emailjs.com/api/v1.0/email/send');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return $httpCode === 200;
}

/**
 * Internal helper to log mail activity
 */
function _logMailActivity($to, $subject, $body, $status, $provider, $error = '') {
    global $pdo;
    try {
        $stmt = $pdo->prepare("INSERT INTO mail_activity (user_id, email_type, recipient_email, status, details, created_at) VALUES (?, ?, ?, ?, ?, datetime('now'))");
        
        $userId = $_SESSION['user_id'] ?? null;
        
        $details = json_encode([
            'subject' => $subject,
            'provider' => $provider,
            'error' => $error,
            // truncated body to save space if needed, or full body
            'snippet' => substr(strip_tags($body), 0, 100)
        ]);

        $stmt->execute([$userId, 'system_email', $to, $status, $details]);
    } catch (Exception $e) {
        // Silent fail for logging
        error_log("Mail logging failed: " . $e->getMessage());
    }
}
