<?php
// backend/tests/test_logic_fixes.php

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../utils/helpers.php';

echo "Starting Logic Fix Verification...\n\n";

// 1. Timezone Check
echo "1. Checking Timezone Consistency...\n";
$phpTime = date('Y-m-d H:i:s');
$stmt = $pdo->query("SELECT datetime('now') as sqlite_time");
$sqliteTime = $stmt->fetch()['sqlite_time'];

echo "PHP UTC Time: $phpTime\n";
echo "SQLite UTC Time: $sqliteTime\n";

if (abs(strtotime($phpTime) - strtotime($sqliteTime)) < 2) {
    echo "✅ Timezone consistency verified (Both UTC).\n\n";
} else {
    echo "❌ Timezone mismatch! (Diff: " . (strtotime($phpTime) - strtotime($sqliteTime)) . "s)\n\n";
}

// 2. Atomic Balance Check (Dry Run)
echo "2. Checking Atomic Balance Logic...\n";
// We can't easily simulate concurrency in this single script without multiple processes,
// but we can verify the SQL logic doesn't allow overdrafts.
try {
    $pdo->beginTransaction();
    
    // Create dummy user and asset
    $userId = generateUUID();
    $pdo->prepare("INSERT INTO users (id, email, password_hash) VALUES (?, ?, ?)")->execute([$userId, 'test@example.com', 'hash']);
    $pdo->prepare("INSERT INTO user_assets (user_id, symbol, name, balance) VALUES (?, 'TEST', 'Test Asset', 10.0)")->execute([$userId]);
    
    // Attempt to debit 15.0 (should fail due to balance >= amount check)
    $stmt = $pdo->prepare("UPDATE user_assets SET balance = balance - ? WHERE user_id = ? AND symbol = ? AND balance >= ?");
    $stmt->execute([15.0, $userId, 'TEST', 15.0]);
    
    if ($stmt->rowCount() === 0) {
        echo "✅ Atomic overdraft protection verified (rowCount 0 for insufficient funds).\n";
    } else {
        echo "❌ Atomic logic error: row updated despite insufficient funds!\n";
    }
    
    $pdo->rollBack();
} catch (Exception $e) {
    echo "❌ Atomic check exception: " . $e->getMessage() . "\n";
    if ($pdo->inTransaction()) $pdo->rollBack();
}
echo "\n";

// 3. Username Uniqueness Check
echo "3. Checking Username Uniqueness...\n";
try {
    $pdo->beginTransaction();
    $userId1 = generateUUID();
    $userId2 = generateUUID();
    
    $pdo->prepare("INSERT INTO users (id, email, password_hash) VALUES (?, ?, ?)")->execute([$userId1, 'test1@example.com', 'hash']);
    $pdo->prepare("INSERT INTO user_profiles (id, user_id, first_name, last_name, username) VALUES (?, ?, 'John', 'Doe', 'johndoe')")->execute([generateUUID(), $userId1]);
    
    // Attempt to create another 'johndoe' (simulating logic from AuthController)
    $baseUsername = 'johndoe';
    $username = $baseUsername;
    $stmt = $pdo->prepare("SELECT id FROM user_profiles WHERE username = ?");
    $stmt->execute([$username]);
    if ($stmt->fetch()) {
        $username = $baseUsername . '_salt'; // simplified for test
    }
    
    if ($username !== 'johndoe') {
        echo "✅ Unique username generation verified ($username).\n";
    } else {
        echo "❌ Unique username generation failed!\n";
    }
    
    $pdo->rollBack();
} catch (Exception $e) {
    echo "❌ Username check exception: " . $e->getMessage() . "\n";
    if ($pdo->inTransaction()) $pdo->rollBack();
}

echo "\nVerification Complete.\n";
